<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

// Get refund ID or sale ID from URL
$refund_id = $_GET['refund_id'] ?? null;
$sale_id = $_GET['sale_id'] ?? null;

if (!$refund_id && !$sale_id) {
    die('Refund ID or Sale ID is required');
}

// If sale_id is provided, get the refund_id for that sale
if ($sale_id && !$refund_id) {
    $stmt = $pdo->prepare("SELECT id FROM refunds WHERE sale_id = ? ORDER BY created_at DESC LIMIT 1");
    $stmt->execute([$sale_id]);
    $refund_id = $stmt->fetchColumn();
    
    if (!$refund_id) {
        die('No refund found for this sale');
    }
}

try {
    // Get store information
    $stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_binary FROM store_settings WHERE setting_group = 'store_info'");
    $stmt->execute();
    $settings = [];
    $logoData = null;
    $logoName = '';
    while ($row = $stmt->fetch()) {
        $settings[$row['setting_key']] = $row['setting_value'];
        if ($row['setting_key'] === 'store_logo') {
            $logoData = $row['setting_binary'];
            $logoName = $row['setting_value'];
        }
    }

    // Debug output to verify store settings
    error_log("Store Settings: " . print_r($settings, true));

    // Get refund details with receipt number
    $stmt = $pdo->prepare("
        SELECT r.*, s.invoice_number, r.refunded_at as refund_date, c.name as customer_name,
               DATE_FORMAT(r.refunded_at, '%Y-%m-%d %H:%i:%s') as refund_date,
               CONCAT('REF', LPAD(r.id, 6, '0')) as refund_receipt_no,
               u.username as processed_by,
               r.amount as refund_amount
        FROM refunds r
        JOIN sales s ON r.sale_id = s.id
        LEFT JOIN customers c ON s.customer_id = c.id
        LEFT JOIN users u ON r.processed_by = u.id
        WHERE r.id = ?
    ");
    $stmt->execute([$refund_id]);
    $refund = $stmt->fetch();

    if (!$refund) {
        die('Refund not found');
    }

    // Get refund items with manual amounts
    $stmt = $pdo->prepare("
        SELECT ri.*, p.name as product_name,
               CASE 
                   WHEN ri.condition IN ('Damaged', 'Opened', 'Other') THEN ri.manual_amount
                   ELSE (ri.unit_price * ri.quantity) - COALESCE(ri.discount, 0)
               END as final_amount
        FROM refund_items ri
        JOIN products p ON ri.product_id = p.id
        WHERE ri.refund_id = ?
    ");
    $stmt->execute([$refund_id]);
    $refund_items = $stmt->fetchAll();

    // Process serial numbers for each item
    foreach ($refund_items as &$item) {
        if (!empty($item['serial_numbers'])) {
            $item['serial_numbers_array'] = json_decode($item['serial_numbers'], true);
            $item['serial_numbers_display'] = implode(', ', $item['serial_numbers_array']);
        } else {
            $item['serial_numbers_array'] = [];
            $item['serial_numbers_display'] = '';
        }
    }

} catch (PDOException $e) {
    die('Error: ' . $e->getMessage());
}

// Get currency from settings
$currency = $settings['currency'] ?? 'Ksh';
$website = $settings['website'] ?? '';

// Store info for header
$store_header = [
    'store_name' => $settings['store_name'] ?? '',
    'store_address' => $settings['store_address'] ?? '',
    'store_phone' => $settings['store_phone'] ?? '',
    'store_email' => $settings['store_email'] ?? '',
    'website' => $settings['website'] ?? ''
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Refund Receipt <?php echo $refund['refund_receipt_no']; ?></title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            margin: 0;
            padding: 20px;
            font-size: 14px;
            background: #f8f9fa;
        }
        .receipt {
            max-width: 80mm;
            margin: 0 auto;
            padding: 15px;
            background: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            border-radius: 8px;
        }
        .header {
            text-align: center;
            margin-bottom: 5px;
            padding-bottom: 3px;
            border-bottom: 1px dashed #000;
        }
        .store-logo {
            max-width: 60mm;
            max-height: 30mm;
            margin: 5px auto;
            display: block;
        }
        .store-name {
            font-size: 20px;
            font-weight: 800;
            margin: 10px 0;
            color: #000;
        }
        .store-details {
            font-size: 13px;
            margin: 5px 0;
            padding-bottom: 10px;
            border-bottom: 1px solid #000;
            color: #000;
        }
        .store-details p {
            margin: 3px 0;
            line-height: 1.4;
        }
        .receipt-header {
            text-align: center;
            margin-bottom: 15px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 6px;
        }
        .receipt-number {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 5px;
            color: #dc3545;
        }
        .receipt-date {
            font-size: 14px;
            margin-bottom: 5px;
            color: #666;
        }
        .client-info {
            font-size: 14px;
            margin-bottom: 5px;
            color: #333;
        }
        .salesperson-info {
            font-size: 14px;
            margin-bottom: 5px;
            color: #333;
        }
        .receipt-title {
            font-size: 18px;
            font-weight: bold;
            text-align: center;
            margin: 15px 0;
            padding: 8px 0;
            border-top: 2px dashed #000;
            border-bottom: 2px dashed #000;
            color: #dc3545;
        }
        .info-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            padding: 4px 0;
        }
        .info-label {
            font-weight: bold;
            color: #333;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 15px 0;
        }
        th, td {
            text-align: left;
            padding: 8px 5px;
            font-size: 12px;
        }
        th {
            border-bottom: 2px solid #000;
            color: #333;
        }
        .item-condition {
            font-size: 11px;
            color: #666;
            padding: 3px 6px;
            background: #f8f9fa;
            border-radius: 4px;
            display: inline-block;
            margin-top: 3px;
            border: 1px solid #dee2e6;
        }
        .manual-amount {
            color: #dc3545;
            font-weight: bold;
        }
        .restock-info {
            font-size: 11px;
            color: #28a745;
            margin-top: 3px;
            padding: 2px 5px;
            background: #e8f5e9;
            border-radius: 3px;
            display: inline-block;
        }
        .total-row {
            border-top: 2px dashed #000;
            margin-top: 15px;
            padding-top: 15px;
        }
        .total-amount {
            font-size: 16px;
            font-weight: bold;
            color: #dc3545;
        }
        .footer {
            text-align: center;
            margin-top: 20px;
            font-size: 12px;
            border-top: 2px dashed #000;
            padding-top: 15px;
            color: #666;
        }
        .processed-by {
            font-size: 11px;
            color: #666;
            text-align: right;
            margin-top: 10px;
        }
        @media print {
            body {
                padding: 0;
                background: white;
            }
            .receipt {
                width: 80mm;
                box-shadow: none;
                border-radius: 0;
            }
            .no-print {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="receipt">
        <div class="header">
            <?php if ($logoData): ?>
                <?php
                $mimeType = 'image/jpeg'; // Default to JPEG
                if (strpos($logoName, '.png') !== false) {
                    $mimeType = 'image/png';
                } elseif (strpos($logoName, '.gif') !== false) {
                    $mimeType = 'image/gif';
                }
                ?>
                <img src="data:<?php echo $mimeType; ?>;base64,<?php echo base64_encode($logoData); ?>" alt="Store Logo" class="store-logo">
            <?php endif; ?>
            <div class="store-name"><?php echo htmlspecialchars($store_header['store_name']); ?></div>
            <div class="store-details">
                <p><?php echo htmlspecialchars($store_header['store_address']); ?></p>
                <p>Phone: <?php echo htmlspecialchars($store_header['store_phone']); ?></p>
                <p>Email: <?php echo htmlspecialchars($store_header['store_email']); ?></p>
                <?php if (!empty($store_header['website'])): ?>
                    <p>Web: <?php echo htmlspecialchars($store_header['website']); ?></p>
                <?php endif; ?>
            </div>
        </div>

        <div class="receipt-header">
            <div class="receipt-number"><?php echo $refund['refund_receipt_no']; ?></div>
            <div class="receipt-date"><?php echo $refund['refund_date']; ?></div>
            <div class="client-info">
                Client: <?php echo htmlspecialchars($refund['customer_name'] ?? 'Walk-in Customer'); ?>
            </div>
            <div class="salesperson-info">
                Cashier: <?php echo htmlspecialchars($refund['processed_by'] ?? 'Unknown'); ?>
            </div>
        </div>

        <div class="receipt-title">RETURN RECEIPT</div>

        <div class="info-row">
            <span class="info-label">Original Sale:</span>
            <span>#<?php echo $refund['invoice_number']; ?></span>
        </div>

        <table>
            <thead>
                <tr>
                    <th>Item</th>
                    <th>Qty</th>
                    <th>Unit Price</th>
                    <th>Total</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($refund_items as $item): ?>
                <tr>
                    <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                    <td><?php echo $item['quantity']; ?></td>
                    <td><?php echo $currency . ' ' . number_format($item['unit_price'], 2); ?></td>
                    <td>
                        <?php if (in_array($item['condition'], ['Damaged', 'Opened', 'Other'])): ?>
                            <span class="manual-amount">
                                <?php echo $currency . ' ' . number_format($item['manual_amount'], 2); ?>
                            </span>
                        <?php else: ?>
                            <?php echo $currency . ' ' . number_format($item['unit_price'] * $item['quantity'], 2); ?>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <td colspan="4">
                        <div class="item-condition">
                            Condition: <?php echo htmlspecialchars($item['condition']); ?>
                            <?php if (in_array($item['condition'], ['Damaged', 'Opened', 'Other'])): ?>
                                (Manual Amount)
                            <?php endif; ?>
                        </div>
                        <?php if ($item['restock_quantity'] > 0): ?>
                            <div class="restock-info">
                                Restocked: <?php echo $item['restock_quantity']; ?> units
                            </div>
                        <?php endif; ?>
                        <?php if (!empty($item['serial_numbers_display'])): ?>
                            <div class="restock-info" style="background: #e3f2fd; color: #1976d2;">
                                Serial Numbers: <?php echo htmlspecialchars($item['serial_numbers_display']); ?>
                            </div>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="total-row">
            <div class="info-row">
                <span class="info-label">Total Refund Amount:</span>
                <span class="total-amount"><?php echo $currency . ' ' . number_format($refund['refund_amount'], 2); ?></span>
            </div>
        </div>

        <div class="processed-by">
            Processed by: <?php echo htmlspecialchars($refund['processed_by'] ?? 'System'); ?>
        </div>

        <div class="footer">
            Thank you for your business!<br>
            <small>Designed and developed by Vico Technologies</small>
        </div>

        <?php if (isset($_GET['print'])): ?>
        <script>
            window.onload = function() {
                window.print();
            };
            window.onafterprint = function() {
                window.close();
                if (window.opener) {
                    window.opener.location.href = 'reports.php';
                } else {
                    window.location.href = 'reports.php';
                }
            };
        </script>
        <?php endif; ?>
    </div>
</body>
</html> 